import importlib
from datetime import datetime

from flask import g


from huansi_utils.server.service_uc import HSSingleUCService
from huansi_utils.celery import celery
from huansi_utils.celery.auth import DefaultAuth
from huansi_utils.db.db import new_session
from huansi_utils.common.json import format_json_data
from .pbInterfaceInvokeLog_schema import PB_InterfaceInvokeLog_schema
from celery.worker.request import Request
from flask_app import global_app
import celery_config


class DataPackage:

    def __init__(self, data_name, key, data, session, destination):
        self.name = data_name
        self.key = key
        self.upload_time = datetime.now()
        self.data = data
        self.from_ = 'mes'
        self.to = destination
        self.session = session

    @property
    def data_package(self):
        '''
        打包数据，并格式化数据
        :return:
        '''
        d = self.__dict__.copy()
        d.pop('session')
        d.update({'from': d.pop('from_')})
        return format_json_data(d)

    def save(self):
        '''
        获取到数据后，写入中间表
        '''
        schema_data = PB_InterfaceInvokeLog_schema().load(self.data_package).data
        # 保存到数据库
        try:
            self.session.begin_trans()
            sql = '''
                INSERT INTO dbo.pbInterfaceInvokeLog
                (sFrom, sTo, sType, sKey, sData, tCreateTime)
                VALUES('{sFrom}', '{sTo}', '{sType}', '{sKey}', '{sData}', '{tCreateTime}')'''.format(**schema_data)
            self.session.exec_sql(sql)
            self.session.commit_trans()
        except Exception:
            self.session.rollback_trans()


class HsRequest(Request):
    '接收到任务先创建一条日志'

    def __init__(self, *args, **kwargs):
        with global_app.app_context():
            self.session = new_session()
        self.header = kwargs['headers']
        self.args = eval(self.header['kwargsrepr'])
        if self.args.get('write_log', True):
            self.write_log()
        super().__init__(*args, **kwargs)

    def write_log(self):
        sTaskKey = self.header['id'] # 获取任务key
        sTask = self.header.get('task')  # 获取任务函数
        for key, value in celery_config.CELERYBEAT_SCHEDULE.items():
            if value['task'] == sTask and str(value['kwargs']) == self.header['kwargsrepr']:
                sTaskName = key # 获取任务名称
                break
        sArg = self.header['kwargsrepr'].replace("'", '"') # 传递的所有参数
        try:
            self.session.begin_trans()
            sql = '''
                INSERT INTO dbo.pbCeleryTaskHistory
                (sTaskName, sTask, sArg, sTaskKey)
                VALUES('{sTaskName}', '{sTask}', '{sArg}', '{sTaskKey}')'''.\
            format(sTaskName=sTaskName,
                   sTask=sTask,
                   sArg=sArg,
                   sTaskKey=sTaskKey)
            self.session.exec_sql(sql)
            self.session.commit_trans()
        except Exception:
            self.session.rollback_trans()


class HsTaskService(celery.Task, HSSingleUCService):
    abstract=True
    Request = HsRequest
    def __init__(self):
        with global_app.app_context():
            self.session = new_session()
            g.hs_session = self.session
            super().__init__()

    def get_auth_class(self, to):
        '''
        向目标方发送数据时获取认证类定义的认证规则，如未获取到则使用默认类
        :return:返回认证类对象
        '''
        auth_config = celery_config.AUTH_CLASS.get(to, None)
        if auth_config is None:
            return DefaultAuth(to)
        module_name, auth_class_name = auth_config.rsplit('.', 1)
        module = importlib.import_module(module_name)
        auth_class = getattr(module, auth_class_name, None)
        if auth_class is None:
            return DefaultAuth(to)
        return auth_class(to)


    def do_task(self, **kwargs):
        key = self.request.id   # 获取消息key，这个key是收到任务后celery生成的
        self.data_type = kwargs.get('type') # 获取传输的数据类型
        to = kwargs.get('to')   # 获取数据发送的目标方
        sql = '''
            declare @sData NVARCHAR(MAX)=0x
            declare @sType INT=0
            EXEC dbo.sppbInterface_Export @sTypeName='{}',@sData=@sData OUTPUT,@sType=@sType OUTPUT
            select @sData, @sType
        '''.format(self.data_type)
        upload_data = self.sql_data_to_dict(self.query_sql(sql)) # 将sql查询出来的数据转成约定好的数据格式
        if isinstance(to, str):
            self.send_data_single(to, key, upload_data)
        elif isinstance(to, list): # 向多个系统发送数据，进行循环发送
            for system in to:
                self.send_data_single(system, key, upload_data)

    def sql_data_to_dict(self, sql_data):
        '''
        将sql查询出来的数据转成约定好的数据格式
        :param sql_data:
        :return:
        '''
        result = self.dump_data(sql_data)
        add, del_ = [], []
        for row in result:
            if row['is_del'] == '1':
                row.pop('is_del')
                del_.append(row)
            else:
                row.pop('is_del')
                add.append(row)
        return {'add': add, 'del': del_}

    def handle_failure(self, exc, to):
        try:
            self.session.begin_trans()
            sql = '''
                UPDATE dbo.pbInterfaceInvokeLog
                SET iExecResult={},sResult='{}',tEndTime='{tEndTime}'
                WHERE sKey='{sKey}' AND sTo='{sTo}'
                '''.format(1, str(exc).replace("'", '"'), sKey=str(self.request.id), sTo=str(to), tEndTime=datetime.now().strftime("%Y-%m-%d %H:%M:%S"))
            self.session.exec_sql(sql)
            self.session.commit_trans()
        except Exception:
            self.session.rollback_trans()

    def send_data_single(self, to, key, upload_data):
        '''
        打包数据，向其他系统接口转发
        :param to: 目标方
        :param key: 消息key
        :param upload_data: 将要上传的数据
        :return:
        '''
        data_package = DataPackage(self.data_type, key, upload_data, self.session, to)
        data_package.save()
        auth_class = self.get_auth_class(to)
        # 如果在请求过程中出现错误，将错误信息写入日志
        try:
            auth_class.send_data(data_package.data_package)
        except Exception as e:
            self.handle_failure(e, to)

    def on_failure(self, exc, task_id, args, kwargs, einfo):
        '''
        任务失败时将错误堆栈信息写到数据库
        :param exc:
        :param task_id:
        :param args:
        :param kwargs:
        :param einfo:
        :return:
        '''
        if kwargs.get('write_log', True):
            try:
                self.session.begin_trans()
                sql = '''
                        UPDATE dbo.pbCeleryTaskHistory
                        SET sErrorMessage='{}', tEndTime='{tEndTime}'
                        WHERE sTaskKey='{}'
                                '''.format(str(einfo).replace("'", '"'), task_id,tEndTime=datetime.now().strftime("%Y-%m-%d %H:%M:%S"))
                self.session.exec_sql(sql)
                self.session.commit_trans()
            except Exception:
                self.session.rollback_trans()
        return super(HsTaskService, self).on_failure(exc, task_id, args, kwargs, einfo)

    def on_success(self, retval, task_id, args, kwargs):
        if kwargs.get('write_log', True):
            try:
                self.session.begin_trans()
                sql = '''
                        UPDATE dbo.pbCeleryTaskHistory
                        SET tEndTime='{tEndTime}'
                        WHERE sTaskKey='{sTaskKey}'
                                '''.format(tEndTime=datetime.now().strftime("%Y-%m-%d %H:%M:%S"), sTaskKey=task_id)
                self.session.exec_sql(sql)
                self.session.commit_trans()
            except Exception:
                self.session.rollback_trans()
        return super(HsTaskService, self).on_success(retval, task_id, args, kwargs)

