import datetime
import math
from queue import Queue
from threading import Thread


class LogCache():
    def __init__(self, cache_count=1000):
        if cache_count < 0:
            cache_count = 0

        self.cache_queue = Queue(cache_count)

    def put(self, **data):
        '''
        往队列put数据（队列溢出将会自动清空队列数据，再put）
        :param data:
        :return:
        '''
        if self.cache_queue.full():
            self.cache_queue.queue.clear()

        self.cache_queue.put(data)


class Logger():
    def log_start(self, **message):
        '''
        开始记录日志
        :param message:
        :return:
        '''
        message['start_time'] = datetime.datetime.now()
        return message

    def log_end(self, **message):
        '''
        结束记录日志
        :param message:
        :return:
        '''
        start_time = message.pop('start_time')
        end_time = datetime.datetime.now()
        diff_seconds = math.floor((end_time - start_time).total_seconds() * 1000)
        message['diff_seconds'] = diff_seconds
        return message


class LogThread():
    def __init__(self, cache_count=1000):
        self.log_cache = LogCache(cache_count)
        self.logger = Logger()

    def handle_message(self, func, queue):
        '''
        消息处理
        :param func:
        :param queue:
        :return:
        '''
        while True:
            try:
                message = queue.get()
                if message:
                    func(**message)
                else:
                    import time
                    time.sleep(0.1)
            except Exception as e:
                print('LogThread函数处理失败：', e)

    def put(self, **message):
        '''
        put数据（记入日志）
        :param message:
        :return:
        '''
        self.log_cache.put(**message)

    def start(self, func):
        '''
        开始执行
        :param func:
        :return:
        '''
        t = Thread(target=self.handle_message, args=(func, self.log_cache.cache_queue))
        t.start()

    def log_start(self, **message):
        return self.logger.log_start(**message)

    def log_end(self, **message):
        return self.logger.log_end(**message)


logThread = LogThread()
