import time
from contextlib import contextmanager

from flask import current_app

from huansi_utils.app.apploader import logger, AppLoaderBase
from huansi_utils.common.cache_log import logThread


@contextmanager
def auto_handle_exception(**kwargs):
    """
    自动处理异常，避免捕获异常代码在多个函数内重复书写
    :param kwargs:
    """
    try:
        proxy_url = None
        # 如果存在代理地址是异步调用的
        if 'proxy_url' in kwargs:
            proxy_url = kwargs.pop('proxy_url')
            yield proxy_url, kwargs
        # 否则是同步调用
        else:
            # 判断是否打开redis记录日志开关
            hsrpc_status = current_app.config.get('HSRPC_STATUS', False)
            # 增加判断如果未配置代理rpc地址
            app_rpc = current_app.config.get('RPC_PROXY_URL')
            if not hsrpc_status or not app_rpc:
                yield
            else:
                # 将tenant_code放到kwargs中
                if not 'tenant_code' in kwargs:
                    kwargs.update({'tenant_code': current_app.config.get('TENANT_CODE', 'huansi')})
                yield proxy_url, kwargs
    except Exception as e:
        logger.error('redis记录日志出错{}'.format(str(e)))


def exec_rpc_request(rpc_func_name, **kwargs):
    """
    真正执行rpc的请求，内部捕获异常
    :param rpc_func_name:
    :param kwargs:
    :return:
    """
    with auto_handle_exception(**kwargs) as proxy_url_and_kwargs:
        if proxy_url_and_kwargs is None:
            return

        app_rpc = proxy_url_and_kwargs[0]
        message = proxy_url_and_kwargs[1]
        if not app_rpc:
            app_rpc = current_app.config.get('RPC_PROXY_URL')

        logThread.put(rpc_func_name=rpc_func_name
                      , message=message
                      , app_rpc=app_rpc
                      , app_code='Tool')

        # response = rpc_request_invoke(rpc_func_name,
        #                               message=proxy_url_and_kwargs[1],
        #                               app_rpc=proxy_url_and_kwargs[0],
        #                               app_code='Tool')
        # return response


def handle_rpc_message(**kwargs):
    '''
    线程内处理rpc消息
    :param kwargs:
    :return:
    '''
    import grpc
    import json
    from hs_rpc.proto.rpc_data_pb2_grpc import GreeterStub
    from hs_rpc.proto.rpc_data_pb2 import Request

    app_code = kwargs.get('app_code')
    app_rpc = kwargs.get('app_rpc')
    message = kwargs.get('message')
    func = kwargs.get('rpc_func_name')

    message.update({'func': func})
    channel = grpc.insecure_channel(app_rpc)
    stub = GreeterStub(channel, app_code=app_code)
    response = stub.handle_request(Request(message=json.dumps(message, ensure_ascii=True)))
    return json.loads(response.message)


def redis_exec(data):
    '''
    redis命令执行---->Tool上的rpc
    本接口可直接执行redis命令，同时需要考虑安全性，使用说明如下
    1. 当请求方为我们自己的服务器或者本地测试，如果通过白名单IP的过滤，本接口可直接执行命令，例如：
        data = {"sql":"set a a&&set b b||get c"}
    2. 或者通过注册的命令id请求，可携带需要替换的参数进行请求,例如:
        data = {"id":"1577329134156-0", "iProjectId":1}
    :param data:需要传入一个对象，
    :return:
    '''
    if not current_app.config.get('HSRPC_STATUS', False):
        return None

    response = rpc_request_invoke('redis_exec',
                                  message={'data': data},
                                  app_code='Tool')
    return response


def redis_log(**kwargs):
    """
    使用redis记录日志
    :param type_: 日志类型
    :param key: 日志键名称
    :param data: 日志内容
    :param source: 日志来源
    :param second: 日志有效期
    :param compress: 是否压缩
    :param max_count: 最大日志行数
    :param tenant_code: 租户code
    :return: 日志id
    """
    return exec_rpc_request('redis_log', **kwargs)


def redis_log_start(**kwargs):
    """
    使用redis记录日志
    :param type_: 日志类型
    :param key: 日志键名称
    :param data: 日志内容
    :param source: 日志来源
    :param second: 日志有效期
    :param compress: 是否压缩
    :param max_count: 最大日志行数
    :param tenant_code: 租户code
    :return: 日志id
    """
    # 本地缓存记录日志，不发送rpc消息，与redis_log_end配套使用
    return logThread.log_start(**kwargs)


def redis_log_end(**kwargs):
    """
    回写日志
    :param type_: 日志类型
    :param key: 日志键名称
    :param log_id: 日志ID
    :param compress: 是否压缩
    :param status: 是否成功(1=成功,-1=失败)
    :param result: 日志记录执行结果/错误信息
    :param tenant_code: 租户code
    :return: 执行结果
    """
    message = logThread.log_end(**kwargs)
    return exec_rpc_request('redis_log_ex', **message)


def redis_log_api_count(**kwargs):
    """
    api的访问计数
    :param key:
    :param api:
    :return:
    """
    return exec_rpc_request('redis_log_api_count', **kwargs)


def redis_log_api_timeout(**kwargs):
    """
    api超时记录
    :param key:
    :param api:
    :param run_total_millisecond:
    :return:
    """
    return exec_rpc_request('redis_log_api_timeout', **kwargs)


def redis_stream_push(**kwargs):
    """
    临时数据推入
    :param data:
    :return:
    """
    return exec_rpc_request('redis_stream_push', **kwargs)


def redis_stream_pop(**kwargs):
    """
    临时数据取出
    :param id:
    :return:
    """
    return exec_rpc_request('redis_stream_pop', **kwargs)


def redis_table_push(**kwargs):
    """
    redis表数据插入
    :param app_code:
    :param table_name:
    :param data:
    :param tenant_code:
    :return: redis流id:
    """
    if 'app_code' not in kwargs:
        app_code = current_app.config.get('APP_CODE')
        if app_code is None:
            from huansi_utils.exception.exception import HSArgumentError
            raise HSArgumentError('未配置环境变量APP_CODE')
        kwargs.update({'app_code': app_code})
    return exec_rpc_request('redis_table_push', **kwargs)


def get_lock(**kwargs):
    """
    获取锁, 此函数使用的大部分场景应该是在app启动时，提供线程锁，因为项目启动时app对象未入栈，所以此时需要传入app对象
    :param tenant_code:
    :param lock_name:
    :param lock_owner:
    :param count:
    :param second:
    :param max_remain_second:
    :param app: app对象，默认None
    :param time_out: 单位：秒，如果传入，将在指定的时间内持续获取锁， 默认None
    :return: 成功获取返回锁的序号， 失败返回-1
    """
    app = kwargs.get('app')
    if app is None and not dir(current_app):
        raise Exception('获取锁失败，脱离上下文，请传入app对象')
    if app:
        # 如果传入了app，有必要先测试一下代理地址是否能联通
        kwargs.pop('app')
        with app.app_context():
            AppLoaderBase().test_connection(current_app)
        if not app.config.get('HSRPC_STATUS', False):
            return
    else:
        if not current_app.config.get('HSRPC_STATUS', False):
            return
    time_out = kwargs.get('time_out')

    def _get_lock(**kwargs):
        logger.info('正在获取{}锁'.format(kwargs.get('lock_name')))
        lock = exec_rpc_request('get_lock', **kwargs)
        if lock:
            if lock['code'] == 200:
                if lock['message'] != -1:
                    return lock['message']
            else:
                raise Exception('获取锁失败，' + lock['message'])

    def _get_timeout_lock(**kwargs):
        total_run_time = 0
        start_time = time.time()
        while total_run_time < time_out:
            r = _get_lock(**kwargs)
            if r:
                return r
            time.sleep(0.1)
            total_run_time = time.time() - start_time

    if time_out:
        kwargs.pop('time_out')
        if app:
            with app.app_context():
                return _get_timeout_lock(**kwargs)
        else:
            return _get_timeout_lock(**kwargs)
    else:
        if app:
            with app.app_context():
                return _get_lock(**kwargs)
        else:
            return _get_lock(**kwargs)


def del_lock(**kwargs):
    """
    释放锁
    :param tenant_code:
    :param lock_name:
    :param lock_id:
    :param lock_owner:
    :return: 成功删除返回1， 失败返回-1
    """
    return exec_rpc_request('del_lock', **kwargs)


def remain_lock(**kwargs):
    """
    保持锁
    :param tenant_code:
    :param lock_name:
    :param lock_id:
    :param lock_owner:
    :param second:
    :return: 成功获取返回1， 失败返回-1
    """
    return exec_rpc_request('remain_lock', **kwargs)
