# -*- coding:utf-8 -*-
import json
import os
import re
import time

from git import GitCommandError

from app.conncetion.conncetion_service import ConnectionService
from app.utils.common_tools import CommonTools
from app.utils.db_tools import db_driver
from flask_app import git_path, repo
from huansi_utils.exception.exception import HSException


class ApplicationService():
    def __init__(self):
        self.init_validate()

    def get_application_list(self, project_no):
        '''
        获取application列表
        :return:
        '''
        self.checkout(project_no)

        compose_content = self.get_compose_content()

        # 获取app列表
        server_app_list = re.findall(r"# ({\"app_code\":.*})", compose_content)

        # 获取本地App列表
        with db_driver as session:
            query_sql = '''select A.*
from app_upgrade_log_dtl A
join app_upgrade_log B on B.id=A.log_id
where B.default_version=1'''

            local_app_list = session.query_sql(query_sql)

        result_list = []

        for server_app in server_app_list:
            app_str = server_app
            server_app = json.loads(server_app)
            server_app['app_str'] = app_str

            server_app_code = server_app['app_code']
            server_app['b_download'] = False

            for local_app in local_app_list:
                local_app_code = local_app['app_code']
                if server_app_code == local_app_code:
                    server_app['b_download'] = True
                    break

            result_list.append(server_app)

        return result_list

    def get_compose_content(self):
        self.compose_path = os.path.join(git_path, 'docker-compose.yml')
        # 读取文件
        with open(self.compose_path, 'r', encoding='utf8') as f:
            compose_content = f.read().replace('\r', '')
        return compose_content

    def checkout(self, project_no):
        try:
            repo.git.checkout(project_no)
            repo.git.pull()
        except GitCommandError as e:
            raise HSException(f'切换分支报错，报错原因:{e},联系技术人员协调解决')

    def init_validate(self):
        '''
        初始化验证
        :return:
        '''
        if not os.path.exists(git_path):
            raise HSException('未能找到Git仓库，联系技术人员协调解决')

    def edit_application_list(self, project_no, json_data_list):
        '''
        修改应用列表并上传git
        :param project_no:
        :param json:
        :return:
        '''
        CommonTools.validate_runner_is_install()
        self.checkout(project_no)

        compose_content = self.get_compose_content()

        # 全选是*
        if json_data_list == '*':
            app_list_str = '*'
        else:
            app_list = []
            for app_str in json_data_list:
                app = re.findall(f'# {app_str} *\n *(.*):', compose_content)
                if not app or len(app) > 1:
                    raise HSException('获取app出错，联系技术人员协调解决')
                app_list.append(app[0])
            app_list_str = ' '.join(app_list)

        compose_content = re.sub('# app_list: "(.*)"', f'# app_list: "{app_list_str}"', compose_content)

        with open(self.compose_path, 'w', encoding='utf8') as f:
            f.write(compose_content)
        # ---------------------
        download_log_path = os.path.join(git_path, 'download_log.log')

        if os.path.exists(download_log_path):
            with open(download_log_path, 'r') as f:
                log_content = f.read() + '\n'
        else:
            with open(download_log_path, 'a+') as f:
                log_content = f.read()

        log_content = f"{log_content}{time.strftime('%Y-%m-%d %H:%M:%S',time.localtime(time.time()))}===>[{app_list_str}]"

        with open(download_log_path, 'w', encoding='utf8') as f:
            f.write(log_content)

        repo.git.add([self.compose_path, download_log_path])
        repo.git.commit(f"-m {time.strftime('%Y-%m-%d %H:%M:%S',time.localtime(time.time()))}===>[{app_list_str}]")
        repo.git.push()

    def get_port_list(self, project_no):
        '''
        获取端口号列表
        :param project_no:
        :return:
        '''
        self.checkout(project_no)
        compose_content = self.get_compose_content()

        # 获取app列表
        app_str_list = re.findall(r"# ({\"app_code\":.*})", compose_content)

        app_list = []
        for app_str in app_str_list:
            _dict = {}
            app_dict = json.loads(app_str, encoding='utf8')
            app_code = app_dict.get('app_code')
            _dict['app_code'] = app_code
            api_port = app_dict.get('api_port')
            if api_port:
                _dict['port'] = api_port
                app_list.append(_dict)

            api_port2 = app_dict.get('api_port2')
            if api_port2:
                _dict['port'] = api_port2
                app_list.append(_dict)

            web_port = app_dict.get('web_port')
            if web_port:
                _dict['port'] = web_port
                app_list.append(_dict)

            web_port2 = app_dict.get('web_port2')
            if web_port2:
                _dict['port'] = web_port2
                app_list.append(_dict)

        return app_list

    def test_port(self, project_no, json_data):
        project_info = ConnectionService().get_project_info(project_no)
        host_ip = project_info['host_ip']
        info_list = []
        for port_info in json_data:
            info_dict = {}
            info_dict['port'] = port_info['port']
            info_dict['app_code'] = port_info['app_code']
            try:
                self.test_connect_port(host_ip, port_info['port'])
                info_dict['success'] = True
            except Exception:
                info_dict['success'] = False
            info_list.append(info_dict)
        return info_list

    import timeout_decorator
    # 超过两秒，基本上没救了，告辞~~~~
    @timeout_decorator.timeout(2)
    def test_connect_port(self, ip, port):
        '''
        检查端口是否开放
        :param ip:
        :param port:
        :return:
        '''
        import socket
        sk = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        if not port:
            port = 80
        try:
            sk.connect((ip, int(port)))
            sk.shutdown(2)
        finally:
            sk.close()
