import math
from datetime import datetime
from functools import wraps


from flask import request, current_app, g
from huansi_utils.rpc_tools.redis_rpc import redis_log_api_count, redis_log_api_timeout


def api_count(func):
    """
    api访问统计计数
    :param func:
    :return:
    """
    @wraps(func)
    def decorated_function(*args, **kwargs):
        app_code = fullfil_condition()
        if app_code is None:
            return func(*args, **kwargs)

        # 获取路径
        api = request.url_rule.rule
        now = datetime.strftime(datetime.now(), '%Y%m%d')
        key = 'log:api_count:{date}:{app_code}'.format(date=now, app_code=app_code)
        tenant_code = get_tenant_code()
        redis_log_api_count(key=key, api=api, tenant_code=tenant_code)
        return func(*args, **kwargs)

    return decorated_function


def api_timeout(func):
    """
    api执行时间统计，如果有执行报错则不会写入
    :param func:
    :return:
    """
    @wraps(func)
    def decorated_function(*args, **kwargs):
        app_code = fullfil_condition()
        if app_code is None:
            return func(*args, **kwargs)

        api = request.url_rule.rule
        start_time = datetime.now()
        r = func(*args, **kwargs)
        end_time = datetime.now()
        run_total_millisecond = math.floor((end_time - start_time).total_seconds() * 1000)

        now = datetime.strftime(end_time, '%Y%m%d')
        key = 'log:api_timeout:{date}:{app_code}'.format(date=now, app_code=app_code)
        tenant_code = get_tenant_code()
        redis_log_api_timeout(key=key, api=api, run_total_millisecond=run_total_millisecond, tenant_code=tenant_code)
        return r

    return decorated_function

def fullfil_condition():
    # 判断是否配置app_code, 未配置直接跳过,否则记录无意义
    app_code = current_app.config.get('APP_CODE')
    # 如果配置的rpc代理连接不上，直接跳过
    hsrpc_status = current_app.config.get('HSRPC_STATUS', False)
    if app_code is not None and hsrpc_status:
        return app_code

def get_tenant_code():
    return g.user_info.get('tenant', 'huansi')