from huansi_utils.bill.register import global_bill_types
from huansi_utils.enum.bill_enum import HSBillOperateType, HSBillEvent, HSBillStatus, HSBillAuditType
from huansi_utils.enum.enum import HSSqlObjectType
from huansi_utils.exception.exception import HSException
from huansi_utils.server.server import HSBaseService, HSBaseTableService


# 通用单据处理Service
class HSGlobalBillService(HSBaseService):
    def __init__(self, session, bill_type):
        super().__init__(session)
        self.bill_type = int(bill_type)
        if not self.bill_type or self.bill_type <= 0:
            raise RuntimeError('单据类型[{}]不存在'.format(bill_type))
        self.service = self.new_service(self.bill_type)
        self.table_name = self.service.table_name
        self.bill_type_info = None
        self.bill_hdr_info = None
        self.operator = None

    # 创建单据类型对应的类实例
    def new_service(self, bill_type):
        try:
            cls = global_bill_types[bill_type]
        except KeyError:
            raise HSException('单据类型[{}]未注册Service', bill_type)
        obj = cls.__new__(cls) if cls else None
        if not obj:
            raise RuntimeError('单据类型[{}]未注册Service', bill_type)
        return obj

    # 查询当前操作人
    def _query_operator(self, operator_id):
        sql = '''SELECT A.sUserName,A.sUserNo,[iUserId]=A.iIden
    FROM dbo.saUser A(NOLOCK)
	WHERE A.iIden=:iUserId'''
        sql = '''SELECT [sUserName]='环思',[sUserNo]='huansi',[iUserId]=:iUserId'''
        self.operator = self.retrive_sql(sql, iUserId=operator_id)

    # 查找单据类型的审核方式
    def _query_bill_type_info(self):
        sql = '''SELECT A.iAuditType,A.bOnlySendByCreator,A.iMaxAuditStep
        FROM dbo.pbBillType A(NOLOCK)
        WHERE A.iIden=:iBillTypeId AND A.bUsable=1 AND A.bArchive=0
        '''
        self.bill_type_info = self.retrive_sql(sql, iBillTypeId=self.bill_type)
        if not self.bill_type_info:
            raise RuntimeError('单据类型[{}]不存在，请检查'.format(self.bill_type))

    # 查找单据头信息
    def _query_bill_hdr_info(self, bill_id):
        # sql = '''SELECT A.sMapDtlTables
        # FROM dbo.pbBillType A(NOLOCK)
        # WHERE A.iIden=:iBillTypeId
        # '''
        # map_dtl_name = self.retrive_sql(sql=sql, iBillTypeId=self.bill_type).sMapDtlTables

        sql = '''SELECT * FROM {} (NOLOCK) WHERE id=:id'''.format(self.table_name)
        hdr_info = self.retrive_sql(sql=sql, id=bill_id)
        return self._columns_filter(hdr_info)

    # 单据头信息过滤
    def _columns_filter(self, hdr_info):
        '''
        单据头信息过滤
        :return:
        '''
        if not hdr_info:
            return None
        self.bill_hdr_info = {}
        # columns = ['sBillNo', 'iBillStatus', 'iCreator', 'iCompanyId', 'iDepartmentId', 'bSystem', 'iAuditType',
        #            'sConfirmMan', 'tConfirmTime', 'sAuditMan', 'tAuditTime', 'iCurrAuditStep',
        #            'iMaxAuditStep', 'iRefBillTypeId', 'iRefBillTableId', 'iRefBillDtlTableId', 'iSourceBillTypeId']
        columns = ['bill_no', 'bill_status', 'creator_id', 'iCompanyId', 'iDepartmentId', 'bSystem', 'iAuditType',
                   'sConfirmMan', 'tConfirmTime', 'auditor_name', 'audit_time', 'iCurrAuditStep',
                   'iMaxAuditStep', 'iRefBillTypeId', 'iRefBillTableId', 'iRefBillDtlTableId', 'iSourceBillTypeId']
        hdr_info = dict(hdr_info)
        for column in columns:
            if hdr_info.get(column) is not None:
                self.bill_hdr_info[column] = hdr_info.get(column)
        return self.bill_hdr_info

    # 回写单据 审核方式 最大审核步骤
    def _write_back_bill(self, bill_id):
        '''
        回写单据 审核方式 最大审核步骤
        :param bill_id:
        :return:
        '''
        update_sql = "iAuditType={}".format(self.bill_type_info.iAuditType)
        if self.bill_hdr_info.get('iMaxAuditStep'):
            update_sql = "{},iMaxAuditStep={}".format(update_sql, self.bill_hdr_info.get('iMaxAuditStep'))
        update_sql = '''UPDATE A
        SET {}
        FROM dbo.{} A
        WHERE A.iIden={}
        '''.format(update_sql, self.table_name, bill_id)
        self.exec_sql(update_sql)

    # 登记单据流程日志
    def _log_bill_flow_history(self, bill_id, operator_id, operate_type, audit_step=-1, message=None, flow_remark=None):
        '''
        登记单据流程日志
        :param bill_id:单据ID
        :param operate_id:操作人
        :param operate_type:单据操作类型
        :param audit_step: 审批步骤(仅适用于多级审批)
        :param message:日志说明
        :param flow_remark:操作时的附加数据(如驳回原因等)
        :return:
        '''
        if self.bill_type <= 0 or bill_id <= 0:
            return
        sql = '''SELECT TOP 1 A.sBillType,A.bLogFlowHistory
	FROM dbo.pbBillType A(NOLOCK)
	WHERE A.iIden=:iBillTypeId
        '''
        bill_type_info = self.retrive_sql(sql, iBillTypeId=self.bill_type)
        # 单据类型，配置为无需记录日志的，直接退出
        if bill_type_info.bLogFlowHistory == 0:
            return
        # 根据操作类型，计算单据状态
        old_status = 0
        new_status = 0
        status = ''
        operate_type = operate_type.value
        if operate_type == HSBillOperateType.SendToAudit.value:
            old_status = 0
            new_status = 1
            status = '送审'
        elif operate_type == HSBillOperateType.Audit.value:
            if audit_step == -1:
                old_status = 1
                new_status = 2
                status = '审核'
            else:  # 审批步骤>=0的审核，则说明是多级审批
                old_status = 1
                new_status = 1
                status = '审批处理'
        elif operate_type == HSBillOperateType.UnAudit.value:
            old_status = 2
            new_status = 0
            status = '取消审核'
        elif operate_type == HSBillOperateType.UnSendToAudit.value:
            old_status = 1
            new_status = 0
            status = '取消送审'
        elif operate_type == HSBillOperateType.BeforeDelete.value:
            old_status = 0
            new_status = 0
            status = '删除'
        elif operate_type == HSBillOperateType.Reject.value:
            old_status = 1
            new_status = 0
            status = '驳回'
        elif operate_type == HSBillOperateType.Print.value:  # (正常情况下，只有审核后的才能打印)
            old_status = 2
            new_status = 2
            status = '打印'
        elif operate_type == HSBillOperateType.Invalid.value:  # (仅草稿单允许作废)
            old_status = 0
            new_status = 5
            status = '作废'
        elif operate_type == HSBillOperateType.Valid.value:
            old_status = 5
            new_status = 0
            status = '取消作废'
        elif operate_type == HSBillOperateType.Close.value:
            old_status = 2
            new_status = 6
            status = '关闭'
        elif operate_type == HSBillOperateType.UnClose.value:
            old_status = 6
            new_status = 2
            status = '取消关闭'
        else:
            raise RuntimeError('单据操作类型[{}]值非法'.format(operate_type))
        if not message:
            message = status
        sql = '''INSERT INTO dbo.pbBillLog
([iBillTypeId],[sBillType],[iBillId],[sBillNo],[iOldStatus],[iNewStatus]
,[iAuditStep],[iOperateType],[sMessage],[iCreator],[sCreator],[sRowNo]
,[tCreateTime],[sFlowRemark])
VALUES(:iBillTypeId,:sBillType,:iBillId,:sBillNo,:iOldStatus,:iNewStatus
,:iAuditStep,:iBillOperateType,:sMessage,:iUserId,:sUserNo,:sRowNo
,GETDATE(),:sFlowRemark)  
        '''
        self.exec_sql(sql, iBillTypeId=self.bill_type, sBillType=bill_type_info.sBillType, iBillId=bill_id,
                      sBillNo=self.bill_hdr_info.get('bill_no')
                      , iOldStatus=old_status, iNewStatus=new_status, iAuditStep=audit_step,
                      iBillOperateType=operate_type
                      , sMessage=message, iUserId=operator_id, sUserNo='', sRowNo='', sFlowRemark=flow_remark)

    # 调用单据类型对应的Service上的事件
    def _call_bill_custom_event(self, bill_id, bill_event):
        sql = '''DECLARE @iResult INT=-1
EXEC dbo.sppbBillOperateEvent @iBillTypeId=:iBillTypeId,@iOperateType=:iOperateType
,@iBillId=:iBillId,@sBillNo=:sBillNo,@iUserId=:iUserId,@sUserNo=:sUserNo
,@iResult=@iResult OUTPUT
SELECT iResult=@iResult
'''
        iResult = self.retrive_sql(sql, iBillTypeId=self.bill_type, iOperateType=bill_event.value, \
                                   iBillId=bill_id, sBillNo=self.bill_hdr_info.get('bill_no') \
                                   , iUserId=self.operator.iUserId, sUserNo=self.operator.sUserNo) \
            .iResult
        if iResult != -1:
            self.db_session.flush()
            return
        # 返回－1说明无存储过程，在代码中处理
        if bill_event == HSBillEvent.SendToAudit:
            self.service.send_bill(bill_id)
        elif bill_event == HSBillEvent.UnSendToAudit:
            self.service.un_send_bill(bill_id)
        elif bill_event == HSBillEvent.Audit:
            self.service.audit_bill(bill_id)
        elif bill_event == HSBillEvent.UnAudit:
            self.service.un_audit_bill(bill_id)
        elif bill_event == HSBillEvent.Validate:
            self.service.validate_bill(bill_id)
        elif bill_event == HSBillEvent.ValidateHint:
            self.service.validate_bill_hint(bill_id)
        elif bill_event == HSBillEvent.CalcTimeStamp:
            self.service.calc_time_stamp(bill_id)
        elif bill_event == HSBillEvent.Close:
            self.service.close_bill(bill_id)
        elif bill_event == HSBillEvent.UnClose:
            self.service.un_close_bill(bill_id)

    # 单据状态转换
    def _bill_status_info(self, bill_status):
        if bill_status == HSBillStatus.Draft.value:
            return '草稿'
        if bill_status == HSBillStatus.Send.value:
            return '送审'
        if bill_status == HSBillStatus.Invalid.value:
            return '作废'
        if bill_status == HSBillStatus.Closed.value:
            return '关闭'
        if bill_status == HSBillStatus.Audit.value:
            return '审核'

    # 更新单据状态
    def _update_bill_status(self, bill_id, operate_type, extend_data=None, in_flow=0):
        '''
        更新单据状态
        :param bill_id:单据ID
        :param operate_type:单据状态
        :param extend_data:扩展数据
        :param in_flow:是否流程中,如果流程中，则状态可改
        :return:
        '''
        updateFiled = ''
        bill_status = self.bill_hdr_info.get('bill_status')
        operate_type = operate_type.value
        if operate_type == HSBillOperateType.SendToAudit.value:
            operate = '送审'
            old_bill_status = 0
            new_bill_status = 1
            updateFiled = ",sConfirmMan='{}',tConfirmTime=GETDATE(),iFlowCount=iFlowCount+1".format(
                self.operator.sUserName)
        elif operate_type == HSBillOperateType.Audit.value:
            operate = '审核'
            old_bill_status = 1
            new_bill_status = 2
            updateFiled = ",sAuditMan='{}',tAuditTime=GETDATE(),iVersion=iVersion+1".format(
                self.operator.sUserName)
        elif operate_type == HSBillOperateType.UnAudit.value:
            operate = '取消审核'
            old_bill_status = 2
            new_bill_status = 0
            updateFiled = ",sConfirmMan='',tConfirmTime=NULL,sAuditMan='',tAuditTime=NULL"
        elif operate_type == HSBillOperateType.UnSendToAudit.value:
            operate = '取消送审'
            old_bill_status = 1
            new_bill_status = 0
            updateFiled = ",sConfirmMan='',tConfirmTime=NULL"
        elif operate_type == HSBillOperateType.Reject.value:
            operate = '驳回'
            old_bill_status = 1
            new_bill_status = 0
            updateFiled = ",sConfirmMan='',tConfirmTime=NULL,sRejectReason='{}' ".format(extend_data)
        elif operate_type == HSBillOperateType.Invalid.value:
            operate = '作废(仅草稿单，允许作废)'
            old_bill_status = 0
            new_bill_status = 5
        elif operate_type == HSBillOperateType.Reject.value + HSBillOperateType.Invalid.value:
            operate = '否决'
            old_bill_status = 1
            new_bill_status = 5
            updateFiled = ",sConfirmMan='',tConfirmTime=NULL,sRejectReason='{}' ".format(extend_data)
        elif operate_type == HSBillOperateType.Valid.value:
            operate = '取消作废'
            old_bill_status = 5
            new_bill_status = 0
        elif operate_type == HSBillOperateType.Close.value:
            operate = '关闭'
            old_bill_status = 2
            new_bill_status = 6
        elif operate_type == HSBillOperateType.UnClose.value:
            operate = '取消关闭'
            old_bill_status = 6
            new_bill_status = 2
        else:
            raise RuntimeError("单据[{}]操作类型[{}]值非法".format(self.bill_hdr_info.get('bill_no'), operate_type))
        # 验证单据原状态是否正确
        if operate_type & 64 > 0:
            # 驳回/否决，允许已提交、已审核时驳回
            if operate_type not in (1, 2):
                raise RuntimeError("单据[{}]状态[{}]未提交/审核,不能驳回".format(self.bill_hdr_info.get('bill_no'), operate_type))
        elif bill_status != old_bill_status and in_flow == 0:
            raise RuntimeError(
                "单据[{}]当前为[{}]状态不等于[{}]状态,不能[{}]".format(self.bill_hdr_info.get('bill_no'),
                                                         self._bill_status_info(self.bill_hdr_info.get('bill_status')),
                                                         self._bill_status_info(old_bill_status),
                                                         operate))
        # sql = '''UPDATE dbo.{}
        # SET bill_status=:bill_status{}
        # WHERE id=:bill_id'''.format(self.table_name, updateFiled)
        sql = '''UPDATE dbo.{} 
        SET bill_status=:bill_status 
        WHERE id=:bill_id'''.format(self.table_name)
        self.exec_sql(sql, bill_status=new_bill_status, bill_id=bill_id)

    # 取消送审日志记录
    def _un_register_bill_flow(self, bill_id, audit_action, audit_step, force=0, message=None, remark=None):
        '''
        从流程中撤回单据
        :param bill_id:wfBill单据ID
        :param audit_action:审批动作
        :param audit_step:审批步骤
        :param force:是否强制完结
        :param message:流程注销原因
        :param remark:流程审批备注
        :return:
        '''
        wfBill_info = self.retrive_sql('''SELECT TOP 1 A.sBillNo,A.iCurrAuditStep
    ,A.sSourceBillType,A.iFlowCount,A.iSourceBillId
    FROM dbo.wfBill A(NOLOCK)
    WHERE A.iIden=@iBillId AND A.iSourceBillTypeId=@iBillTypeId AND A.iFlowCount=1 AND A.bFinished=0''')
        current_audit_step = wfBill_info.get('iCurrAuditStep', -1)
        if current_audit_step == -1:
            raise RuntimeError('当前单据[{}]还未注册到流程，不能撤回'.format(self.bill_hdr_info.get('sBillNo')))
        if current_audit_step != 0 and force == 0:
            raise RuntimeError('当前单据[{}]在流程中或已处理，不能撤回'.format(self.bill_hdr_info.get('sBillNo')))
        # 完结流程
        self.exec_sql('''	UPDATE dbo.wfBill
	SET bFinished=1, sRemark=:sRemark
	WHERE iIden=:iBillId''', sRemark=remark, iBillId=bill_id)
        # 取消送审
        pass  # 暂时不做

    # 校验　单据数据权限
    def _validate_bill_right(self, bill_id_list, dest_bill_right):
        if not bill_id_list:
            return
        # dest_bill_right:更新=1,删除=2,预览=4,打印=8,导出=16,关闭=32,审核=64
        # 是否启用数据权限
        bill_data_right = self.retrive_sql('''SELECT A.bBillDataRight
	FROM dbo.pbBillType A(NOLOCK)
	WHERE A.iIden=:iBillTypeId AND A.bUsable=1 '''.format(self.bill_type)).get('bBillDataRight', 0)
        # 未启用单据数据权限，校验结束
        if bill_data_right == 0:
            return
        # 循环单据ID，逐个单据校验
        bill_id_list = bill_id_list.split_string(',')
        for bil_id in bill_id_list:
            # 查询当前单据的数据权限
            bill_right = self._query_bill_data_right(bil_id)
            # 没权限，报错(位异或之后得到的结果如果在目标权限项中，说明没权限)
            bill_right = (dest_bill_right ^ bill_right) & dest_bill_right
            if bill_right > 0:
                bill_right = self.retrive_sql('''SELECT [sBillRight]=dbo.fnpbConcatString(A.sName)
				FROM dbo.vwpbConst A
				WHERE A.iConstId=286 AND CONVERT(BIGINT,A.sValue) & :iBillRight >0''', iBillRight=bill_right).get(
                    'sBillRight', '')
                if bill_right != '':
                    raise RuntimeError('当前用户[{}]没有单据[{}]的[{}]权限'.format(self.operator, bil_id, bill_right))

    # 查询　单据数据权限
    def _query_bill_data_right(self, bill_id):
        # 是否主档
        archive = self.retrive_sql('''SELECT A.bArchive,A.bBillDataRight
	FROM dbo.pbBillType A(NOLOCK)
	WHERE A.iIden=:iBillTypeId AND A.bUsable=1'''.format(iBillTypeId=self.bill_type))
        # 未启用单据数据权限，则返回目标权限所有权限的和值
        if archive.get('bBillDataRight', 0) == 0:
            return 32767
        # 查找单据的创建人、所在部门
        if archive.get('bArchive', 0) == 0:
            # 查询单据头信息
            pass

    # 数据校验(送审前调用)
    def validate_bill(self, bill_id):
        if not bill_id or bill_id <= 0:
            return
        self._call_bill_custom_event(bill_id, HSBillEvent.Validate)

    # 送审
    def send_bill(self, bill_id, operator_id=0):
        if not bill_id or bill_id <= 0:
            return
        # 判断有无送审过程
        if self.exists_sql_object('sppbBillOperate_SendToAudit', HSSqlObjectType.Script):
            sql = 'EXEC dbo.sppbBillOperate_SendToAudit @iBillTypeId=:bill_type,@iBillId=:bill_id,@iUserId=:user_id'
            self.exec_sql(sql, bill_type=self.bill_type, bill_id=bill_id, user_id=operator_id)
            return
        # 查询当前操作人
        if not self.operator:
            self._query_operator(operator_id)
        # 查找单据类型的审核方式
        self._query_bill_type_info()
        # 查找单据头信息
        self._query_bill_hdr_info(bill_id)
        # 校验当前送审人是否＝单据创建人(系统单除外)
        if not self.bill_hdr_info.get('bSystem') and self.bill_hdr_info.get('creator_id'):
            if self.bill_type_info.bOnlySendByCreator and operator_id != self.bill_hdr_info.get('creator_id'):
                raise RuntimeError(
                    '当前用户[{}]不是单据[{}]的创建人[{}]，不能送审'.format(operator_id, self.bill_hdr_info.get('bill_no'),
                                                           self.bill_hdr_info.get('creator_id')))

        # 调用单据数据校验事件
        # self.valid_bill(bill_id)
        self._call_bill_custom_event(bill_id, HSBillEvent.Validate)
        # 修改单据状态
        self._update_bill_status(bill_id, HSBillOperateType.SendToAudit)
        audit_type = -1
        if self.bill_type_info.iAuditType == HSBillAuditType.Custom.value:
            # 单据审核方式为自定义的，需要在送审过程内部返回最终的审核方式
            audit_type = self._call_bill_custom_event(bill_id, HSBillEvent.Send_To_Audit)
            if not audit_type:
                # 查询单据上的审核方式
                audit_type = self.bill_hdr_info.iAuditType
            if audit_type not in [HSBillAuditType.Auto, HSBillAuditType.Flow, HSBillAuditType.Hand]:
                raise RuntimeError('单据[{}]送审时未指定[审核方式]', self.bill_hdr_info.get('bill_no'))
        else:
            self._call_bill_custom_event(bill_id, HSBillEvent.SendToAudit)

        # 回写单据 审核方式 最大审核步骤
        # self._write_back_bill(bill_id)

        # 登记流程日志
        self._log_bill_flow_history(bill_id=bill_id, operator_id=operator_id,
                                    operate_type=HSBillEvent.SendToAudit)

        if audit_type == HSBillAuditType.Auto.value:
            # 若是自动审核，则调用审核函数
            self.audit_bill(bill_id, operator_id)
        elif audit_type == HSBillAuditType.Flow.value:
            # 若是流程审核，则注册到审批流程中
            self._register_bill_to_flow(bill_id, operator_id)

        # 测试模式下，单据数据校验
        # test_mode = HSSystemVarService().get_system_var(var_group='HS', var_name='test_mode')
        # if test_mode:
        #     self._bill_data_global_validate(bill_id, operator_id)

        # self.commit_trans()

    # 取消送审
    def un_send_bill(self, bill_id, operator_id=0):
        # 判断有无取消送审过程
        if self.exists_sql_object('sppbBillOperate_UnSendToAudit', HSSqlObjectType.Script):
            sql = 'EXEC dbo.sppbBillOperate_UnSendToAudit @iBillTypeId=:bill_type,@iBillId=:bill_id,@iUserId=:user_id'
            self.exec_sql(sql, bill_type=self.bill_type, bill_id=bill_id, user_id=operator_id)
            return
        flow_audit = 0
        # 查询当前操作人
        if not self.operator:
            self._query_operator(operator_id)
        # 查找单据类型的审核方式
        self._query_bill_type_info()
        # 单据表上的审核方式
        audit_type = self.bill_type_info.iAuditType
        # 查找单据头信息
        self._query_bill_hdr_info(bill_id)
        if self.bill_type_info.iAuditType == HSBillAuditType.Custom.value:
            # 查询单据上的审核方式
            audit_type = self.bill_hdr_info.iAuditType
            if audit_type not in [HSBillAuditType.Auto, HSBillAuditType.Flow, HSBillAuditType.Hand]:
                raise RuntimeError('单据[{}]送审时未指定[审核方式]'.format(self.bill_hdr_info.get('bill_no')))
        # 如果是流程审核
        if audit_type == 1:
            # 暂时不做
            raise RuntimeError('暂未开放')
            wfBill_info_finish = self.retrive_sql(
                "SELECT TOP 1 [a]=1 FROM dbo.wfBill WITH(NOLOCK) WHERE iSourceBillId=:iBillId AND iSourceBillTypeId=:iBillTypeId AND iFlowCount=1 AND bFinished=1",
                bill_id, self.bill_type)
            if wfBill_info_finish:
                raise RuntimeError('单据的流程已完结,不能取消送审')
            if not self.retrive_sql(
                    "SELECT TOP 1 [a]=1 FROM dbo.wfBill WITH(NOLOCK) WHERE iSourceBillId=:iBillId AND iSourceBillTypeId=:iBillTypeId AND iFlowCount=1 AND bFinished=0",
                    bill_id, self.bill_type):
                raise RuntimeError('单据未送审,不能取消送审')
            flow_audit = 1
        # 修改单据状态
        self._update_bill_status(bill_id, HSBillOperateType.UnSendToAudit, in_flow=flow_audit)
        # 查询单据信息
        bill_status = self.bill_hdr_info.get('bill_status')
        # 从工作流中查找单据的当前步骤
        # wfBill_info = self.retrive_sql(
        #     "SELECT TOP 1 A.iCurrAuditStep,A.sBillNo,A.iIden,A.sSourceBillType FROM dbo.wfBill WITH(NOLOCK) WHERE iSourceBillId=:iBillId AND iSourceBillTypeId=:iBillTypeId AND iFlowCount=1",
        #     bill_id, self.bill_type)
        if audit_type == 1:  # and wfBill_info:
            # 暂时不做
            raise RuntimeError('暂未开放')
            if wfBill_info_finish:
                raise RuntimeError('流程中的单据[{}]已经完结，不允许取消。'.format(self.bill_hdr_info.get('sBillNo')))
            # 如果是已审单，则调用取消审核
            if bill_status == HSBillStatus.Audit:
                self.un_audit_bill(bill_id, operator_id)
                # 写取消送审日志
                self._un_register_bill_flow(bill_id=wfBill_info.get('iIden'), message='取消审核,流程结束', audit_action=128,
                                            audit_step=wfBill_info.get('iCurrAuditStep'))
            else:
                self._call_bill_custom_event(bill_id, HSBillEvent.UnSendToAudit)
                # 写取消送审日志
                self._un_register_bill_flow(bill_id=wfBill_info.get('iIden'), message='取消审核,流程结束', audit_action=64,
                                            audit_step=wfBill_info.get('iCurrAuditStep'), force=1)
            # 测试模式下，单据数据校验
            test_mode = HSSystemVarService().get_system_var(var_group='HS', var_name='test_mode')
            if test_mode:
                self._bill_data_global_validate(bill_id, operator_id)
        else:
            # 调用取消送审事件
            self._call_bill_custom_event(bill_id, HSBillEvent.UnSendToAudit)
            # 登记流程日志
            self._log_bill_flow_history(bill_id=bill_id, operator_id=operator_id,
                                        operate_type=HSBillEvent.UnSendToAudit)

    # 审核
    def audit_bill(self, bill_id, operator_id=0, validate_bill_right=False):
        # 判断有无审核过程
        if self.exists_sql_object('sppbBillOperate_Audit', HSSqlObjectType.Script):
            sql = 'EXEC dbo.sppbBillOperate_Audit @iBillTypeId=:bill_type,@iBillId=:bill_id,@iUserId=:user_id'
            self.exec_sql(sql, bill_type=self.bill_type, bill_id=bill_id, user_id=operator_id)
            return
        # 查询当前操作人
        if not self.operator:
            self._query_operator(operator_id)
        # 查找单据状态
        self._query_bill_hdr_info(bill_id)
        # 已经审核了的，不再审核(单据可能送审时自动审核了，所以再审核时不报错，而是直接退出)
        if self.bill_hdr_info.get('bill_status') == 2:
            return
        # 多级审批下，必须审批到最后一步才能审核
        if self.bill_hdr_info.get('iMaxAuditStep', 0) > 0 \
                and self.bill_hdr_info.get('iMaxAuditStep', 0) != self.bill_hdr_info.get('iCurrAuditStep', 0):
            raise RuntimeError('当前单据[{}]配置了多级审批，不能直接审核'.format(self.bill_hdr_info.get('sBillNo')))
        # 校验　单据数据权限
        if validate_bill_right == 1:
            return  # 未开放
            self._validate_bill_right(bill_id, dest_bill_right=64)
        # try:
        #     self.begin_trans()
        # 修改单据状态
        self._update_bill_status(bill_id, HSBillOperateType.Audit)
        # 调用审核事件
        self._call_bill_custom_event(bill_id, HSBillEvent.Audit)
        # 还有一处单据流未做

        # 登记流程日志
        self._log_bill_flow_history(bill_id=bill_id, operator_id=operator_id,
                                    operate_type=HSBillEvent.Audit)
        # 测试模式下，单据数据校验
        # test_mode = HSSystemVarService().get_system_var(var_group='HS', var_name='test_mode')
        # if test_mode:
        #     self._bill_data_global_validate(bill_id, operator_id)
        # self.commit_trans()
        # except Exception as e:
        #     self.rollback_trans()
        #     print(e)
        #     raise

    # 取消审核
    def un_audit_bill(self, bill_id, operator_id=0, validate_bill_right=0):
        # 判断有无反审过程
        if self.exists_sql_object('sppbBillOperate_UnAudit', HSSqlObjectType.Script):
            sql = 'EXEC dbo.sppbBillOperate_UnAudit @iBillTypeId=:bill_type,@iBillId=:bill_id,@iUserId=:user_id'
            self.exec_sql(sql, bill_type=self.bill_type, bill_id=bill_id, user_id=operator_id)
            return
        # 查询当前操作人
        if not self.operator:
            self._query_operator(operator_id)
        # 查找单据状态
        self._query_bill_hdr_info(bill_id)
        # 被引用，则不能取消审核
        # self._validate_bill_refedBy(bill_id, message='不能取消审核')
        if validate_bill_right:
            self._validate_bill_right(bill_id, 64)
        # 修改单据状态
        self._update_bill_status(bill_id, HSBillOperateType.UnAudit)
        # 调用取消审核事件
        self._call_bill_custom_event(bill_id, HSBillEvent.UnAudit)
        # 登记流程日志
        self._log_bill_flow_history(bill_id=bill_id, operator_id=operator_id,
                                    operate_type=HSBillEvent.UnAudit)
        # 测试模式下，单据数据校验
        # test_mode = HSSystemVarService().get_system_var(var_group='HS', var_name='test_mode')
        # if test_mode:
        #     self._bill_data_global_validate(bill_id, operator_id)
        # self.commit_trans()

    # 关闭单据(完结)
    def close_bill(self, bill_id, operator_id=0, validate_bill_right=False):
        # 判断有无关闭单据(完结)过程
        if self.exists_sql_object('sppbBillOperate_Close', HSSqlObjectType.Script):
            sql = 'EXEC dbo.sppbBillOperate_Close @iBillTypeId=:bill_type,@iBillId=:bill_id,@iUserId=:user_id'
            self.exec_sql(sql, bill_type=self.bill_type, bill_id=bill_id, user_id=operator_id)
            return
        bill_info = self.retrive_sql('''SELECT TOP 1 A.iEventVersion,A.iBillEventType
        ,[sHdrTableName]=A.sMapTable,[sDtlTableName]=A.sMapDtlTable
        FROM dbo.pbBillType A(NOLOCK)
        WHERE A.iIden=:iBillTypeId AND A.bUsable=1''', iBillTypeId=self.bill_type)
        if not bill_info:
            raise RuntimeError('单据类型[{}]不存在或不可用'.format(self.bill_type))
        event_version = ''
        if bill_info.iEventVersion and bill_info.iEventVersion > 0:
            event_version = '_' + bill_info.iEventVersion
        # 校验单据权限
        if validate_bill_right:
            pass
        # 查询当前操作人
        if not self.operator:
            self._query_operator(operator_id)
        # 查找单据状态
        self._query_bill_hdr_info(bill_id)
        # 修改单据状态为已关闭
        self._update_bill_status(bill_id, HSBillOperateType.Close)
        # 调用整单关闭事件
        if bill_info.iBillEventType & 1024 > 0:
            self._call_bill_custom_event(bill_id, HSBillEvent.Close)
        # 登记流程日志
        self._log_bill_flow_history(bill_id, operator_id, HSBillOperateType.Close)
        # self.commit_trans()

    # 关闭单据明细
    def close_bill_dtl(self, bill_id_list, operator_id=0, ):
        if not bill_id_list:
            return
        # 查询单据类型 配置
        bill_info = self.retrive_sql('''SELECT TOP 1 A.iBillEventType,A.iEventVersion
	FROM dbo.pbBillType A(NOLOCK)
	WHERE A.iIden=:iBillTypeId AND A.bUsable=1''', iBillTypeId=self.bill_type)
        if not bill_info:
            raise RuntimeError('单据类型[{}]不存在或不可用'.format(self.bill_type))
        # 更新明细为已关闭状态
        self.exec_sql('''UPDATE A
	SET iRowStatus=6
	FROM dbo.{} A
	WHERE A.id IN ({}) AND A.iRowStatus=0'''.format(self.table_name, bill_id_list))
        event_version = ''
        if bill_info.get('iEventVersion'):
            event_version = '_' + bill_info.get('iEventVersion')
        self._call_bill_custom_event(bill_id_list, HSBillEvent.Close)
        # 登记流程日志
        self._log_key_data_update(bill_id_list, operator_id, HSBillOperateType.Close)

    # 取消关闭单据
    def un_close_bill(self, bill_id, operator_id=0, validate_bill_right=False):
        # 判断有无取消关闭单据过程
        if self.exists_sql_object('UnClose_Bill', HSSqlObjectType.Script):
            sql = 'EXEC dbo.UnClose_Bill @iBillTypeId=:bill_type,@iBillId=:bill_id,@iUserId=:user_id'
            self.exec_sql(sql, bill_type=self.bill_type, bill_id=bill_id, user_id=operator_id)
            return
        if not bill_id:
            return
        bill_info = self.retrive_sql('''SELECT TOP 1 A.iEventVersion,A.iBillEventType
        ,[sHdrTableName]=A.sMapTable,[sDtlTableName]=A.sMapDtlTable
        FROM dbo.pbBillType A(NOLOCK)
        WHERE A.iIden=:iBillTypeId AND A.bUsable=1''', iBillTypeId=self.bill_type)
        if not bill_info:
            raise RuntimeError('单据类型[{}]不存在或不可用'.format(self.bill_type))
        event_version = ''
        if bill_info.iEventVersion and bill_info.iEventVersion > 0:
            event_version = '_' + bill_info.get('iEventVersion')
        # 校验单据权限，是否有关闭权限
        if validate_bill_right:
            pass
        # 查询当前操作人
        if not self.operator:
            self._query_operator(operator_id)
        # 查找单据状态
        self._query_bill_hdr_info(bill_id)
        # 修改单据状态为已审核
        self._update_bill_status(bill_id, HSBillOperateType.UnClose)
        if bill_info.iBillEventType & 1024 > 0:
            self._call_bill_custom_event(bill_id, HSBillEvent.UnClose)
        # 登记流程日志
        self._update_bill_status(bill_id, HSBillOperateType.UnClose)
        # self.commit_trans()

    # 取消关闭单据明细
    def un_close_bill_dtl(self, bill_id_list, operator_id=0):
        if not bill_id_list:
            return
        # 查询单据类型 配置
        bill_info = self.retrive_sql('''SELECT TOP 1 A.iBillEventType,A.iEventVersion
        	FROM dbo.pbBillType A(NOLOCK)
        	WHERE A.iIden=:iBillTypeId AND A.bUsable=1''', iBillTypeId=self.bill_type)
        if not bill_info:
            raise RuntimeError('单据类型[{}]不存在或不可用'.format(self.bill_type))
        # 更新明细为正常状态
        self.exec_sql('''UPDATE A
        	SET iRowStatus=0
        	FROM dbo.{} A
        	WHERE A.iIden IN ({}) AND A.iRowStatus=6'''.format(self.table_name, bill_id_list))
        event_version = ''
        if bill_info.get('iEventVersion'):
            event_version = '_' + bill_info.get('iEventVersion')
        self._call_bill_custom_event(bill_id_list, HSBillEvent.UnClose)
        # 登记流程日志
        self._log_key_data_update(bill_id_list, operator_id, HSBillOperateType.UnClose)

    # 作废单据
    def invalid_bill(self, bill_id, operator_id=0):
        # 判断有无作废单据过程
        if self.exists_sql_object('sppbBillOperate_Invalid', HSSqlObjectType.Script):
            sql = 'EXEC dbo.sppbBillOperate_Invalid @iBillTypeId=:bill_type,@iBillId=:bill_id,@iUserId=:user_id'
            self.exec_sql(sql, bill_type=self.bill_type, bill_id=bill_id, user_id=operator_id)
            return
        # 查询当前操作人
        if not self.operator:
            self._query_operator(operator_id)
        # 查找单据状态
        self._query_bill_hdr_info(bill_id)
        # 修改单据状态为作废
        self._update_bill_status(bill_id, HSBillOperateType.Invalid)
        # 调用删除前事件(作废等同于删除)
        self._call_bill_custom_event(bill_id, HSBillOperateType.Invalid)
        # 删除单据引用
        # self.exec_sql("DELETE FROM dbo.pbBillRefBill WHERE iBillTypeId=:iBillTypeId AND iBillId=:iBillId",
        #               iBillTypeId=self.bill_type, iBillId=bill_id)
        # 登记流程日志
        self._log_bill_flow_history(bill_id, operator_id, HSBillOperateType.Invalid)
        # self.commit_trans()

    # 恢复作废单据为草稿
    def valid_bill(self, bill_id, operator_id=0):
        # 判断有无恢复作废单据为草稿过程
        if self.exists_sql_object('sppbBillOperate_Valid', HSSqlObjectType.Script):
            sql = 'EXEC dbo.sppbBillOperate_Valid @iBillTypeId=:bill_type,@iBillId=:bill_id,@iUserId=:user_id'
            self.exec_sql(sql, bill_type=self.bill_type, bill_id=bill_id, user_id=operator_id)
            return
        # 查询当前操作人
        if not self.operator:
            self._query_operator(operator_id)
        # 查找单据状态
        self._query_bill_hdr_info(bill_id)
        # 查询单据状态
        self._query_bill_hdr_info(bill_id)
        if self.bill_hdr_info.get('bill_status') != 5:
            raise RuntimeError('单据[{}]未作废，无需恢复'.format(self.bill_hdr_info.get('bill_no')))
        # 修改单据状态(恢复成草稿单)
        self._update_bill_status(bill_id, HSBillOperateType.Valid)
        # 重新计算引用关系
        self._call_bill_custom_event(bill_id, HSBillOperateType.Valid)
        # 登记流程日志
        self._log_bill_flow_history(bill_id, operator_id, HSBillOperateType.Valid)
        # self.commit_trans()

    # 批量 单据校验
    def validate_bills(self, bill_type, bill_ids):
        if not bill_type or bill_type <= 0 or not bill_ids:
            return 500
        try:
            for bill_id in bill_ids:
                self.validate_bill(bill_type, bill_id)
        except Exception as e:
            print(e)
            raise
        return 200

    # 批量删除前校验
    def before_delete_hint(self, bill_ids, operator_id):
        if not bill_ids or bill_ids == '0':
            raise RuntimeError('请传入单据id')
        if isinstance(bill_ids, list):
            _bill_ids = bill_ids
            bill_ids = ','.join(_bill_ids)
        sql = ''' 
        DECLARE @sHint NVARCHAR(200)=0x 
        DECLARE @iHintType INT=0
        EXEC dbo.sppbBillOperate_BeforeDeleteHint @iBillTypeId='{}',@sBillId='{}',@iUserId='{}'
            ,@sHint=@sHint OUTPUT,@iHintType=@iHintType OUTPUT
        SELECT [sHint]=@sHint,[iHintType]=@iHintType''' \
            .format(self.bill_type, bill_ids, operator_id)
        data = self.retrive_sql(sql=sql)
        if data:
            return {'sHint': data.sHint, 'iHintType': data.iHintType}

        # 批量送审
        def send_bills(self, bill_ids, operator_id):
            if not bill_ids or bill_ids == '0':
                raise RuntimeError('请传入单据id')
            if isinstance(bill_ids, list):
                for str in bill_ids:
                    bill_id = int(str)
                    self.send_bill(bill_id, operator_id)
            else:
                bill_id = int(str)
                self.send_bill(bill_id, operator_id)
            return None

        # 批量取消送审
        def un_send_bills(self, bill_ids, operator_id):
            if not bill_ids or bill_ids == '0':
                raise RuntimeError('请传入单据id')
            if isinstance(bill_ids, list):
                for str in bill_ids:
                    bill_id = int(str)
                    self.un_send_bill(bill_id, operator_id)
            else:
                bill_id = int(bill_ids)
                self.un_send_bill(bill_id, operator_id)
            return None

        # 批量审核
        def audit_bills(self, bill_ids, operator_id):
            if not bill_ids or bill_ids == '0':
                raise RuntimeError('请传入单据id')
            if isinstance(bill_ids, list):
                for str in bill_ids:
                    bill_id = int(str)
                    self.audit_bill(bill_id, operator_id)
            else:
                bill_id = int(bill_ids)
                self.audit_bill(bill_id, operator_id)
            return None

        # 批量取消审核
        def un_audit_bills(self, bill_ids, operator_id):
            if not bill_ids or bill_ids == '0':
                raise RuntimeError('请传入单据id')
            if isinstance(bill_ids, list):
                for str in bill_ids:
                    bill_id = int(str)
                    self.un_audit_bill(bill_id, operator_id)
            else:
                bill_id = int(bill_ids)
                self.un_audit_bill(bill_id, operator_id)
            return None

        # 批量关闭单据(完结)
        def close_bills(self, bill_ids, operator_id):
            if not bill_ids or bill_ids == '0':
                raise RuntimeError('请传入单据id')
            if isinstance(bill_ids, list):
                for str in bill_ids:
                    bill_id = int(str)
                    self.close_bill(bill_id, operator_id)
            else:
                bill_id = int(bill_ids)
                self.close_bill(bill_id, operator_id)
            return None

        # 批量取消关闭单据
        def un_close_bills(self, bill_ids, operator_id):
            if not bill_ids or bill_ids == '0':
                raise RuntimeError('请传入单据id')
            if isinstance(bill_ids, list):
                for str in bill_ids:
                    bill_id = int(str)
                    self.un_close_bill(bill_id, operator_id)
            else:
                bill_id = int(bill_ids)
                self.un_close_bill(bill_id, operator_id)
            return None

        # 批量作废单据
        def invalid_bills(self, bill_ids, operator_id):
            if not bill_ids or bill_ids == '0':
                raise RuntimeError('请传入单据id')
            if isinstance(bill_ids, list):
                for str in bill_ids:
                    bill_id = int(str)
                    self.invalid_bill(bill_id, operator_id)
            else:
                bill_id = int(bill_ids)
                self.invalid_bill(bill_id, operator_id)
            return None

        # 批量恢复作废单据为草稿
        def valid_bills(self, bill_ids, operator_id):
            if not bill_ids or bill_ids == '0':
                raise RuntimeError('请传入单据id')
            if isinstance(bill_ids, list):
                for str in bill_ids:
                    bill_id = int(str)
                    self.valid_bill(bill_id, operator_id)
            else:
                bill_id = int(bill_ids)
                self.valid_bill(bill_id, operator_id)
            return None

        # 计算单据时间戳
        def _calc_bill_time_stamp(self, bill_type, bill_ids):
            print('calc_bill_time_stamp')

        # 批量计算单据时间戳
        def _calc_bill_list_time_stamp(self, bill_type, bill_ids):
            if not bill_type or bill_type <= 0 or not bill_ids:
                return
            try:
                for str in bill_ids:
                    bill_id = str if str is int else 0
                    self._calc_bill_time_stamp(bill_type, bill_id)
            except Exception as e:
                print(e)
                raise

    class HSBaseBillDtlService(HSBaseTableService):
        def __init__(self):
            super().__init__()
            self.bill_type = 0
            self.bill_id = 0

        # 关闭单据明细(完结)
        def close_bill_dtl(self, id):
            print('close_bill_dtl')

        # 取消关闭单据
        def un_close_bill_dtl(self, id):
            print('un_close_bill_dtl')
