import redis
from datetime import datetime


class RedisCache(object):
    """ Redis操作函数 """

    def connect(self, host, port, db, password):
        pool = redis.ConnectionPool(host=host,
                                    port=port,
                                    db=db,
                                    password=password,
                                    decode_responses=True)
        self.r = redis.Redis(connection_pool=pool)
        self.pipe = self.r.pipeline()

    def set(self, key, value, expired=None):
        if expired and isinstance(expired, datetime):
            dif = expired - datetime.now()
            seconds = int(dif.total_seconds())
            if seconds < 0:
                seconds = 0
            expired = seconds
        elif isinstance(expired, int):
            pass
        else:
            expired = None

        self.r.set(key, value, ex=expired)

    def mset(self, *args, **kwargs):
        '''
        批量设置值
        r.mset(name1='zhangsan', name2='lisi')
        或
        r.mset({"name1":'zhangsan', "name2":'lisi'})
        :param key:
        :param value:
        :return:
        '''
        self.r.mset(*args, **kwargs)

    def get(self, key):
        '''
        获取值
        :param key:
        :return:
        '''
        return self.r.get(key)

    def mget(self, keys, *args):
        '''
        #批量获取
        r.mget("name1","name2")
        #或
        li=["name1","name2"]
        r.mget(li)
        :param key:
        :param value:
        :return:
        '''
        return self.r.mget(keys, *args)

    def getset(self, key, value):
        '''
        设置新值，打印旧值
        :param key:
        :param value:
        :return:
        '''
        return self.r.getset(key, value)

    def ttl(self, key):
        '''
        返回key到期之前的秒数
        :param key:
        :return:
        '''
        return self.r.ttl(key)

    def expire(self, key, time):
        '''
        给key设置一个到期的秒数
        :param key:
        :param time:
        :return:
        '''
        return self.r.expire(key, time)

    def keys(self, pattern='*'):
        '''
        返回与``pattern``匹配的键列表
        :param pattern:
        :return:
        '''
        return self.r.keys(pattern)

    def flushdb(self):
        '''
        清空数据库
        :return:
        '''
        return self.r.flushdb()

    def get_key_value(self, pattern='*'):
        '''
        返回与``pattern``匹配的键值对
        :param pattern:
        :return:
        '''
        _dict = {}
        keys = self.keys(pattern)
        for item in keys:
            self.pipe.get(item)
        for key, value in zip(keys, self.pipe.execute()):
            _dict[key] = value
        return _dict

    def sadd(self, name, value):
        return self.r.sadd(name, value)

    def smembers(self, name):
        return self.r.smembers(name)

    def delete(self, name):
        self.r.delete(name)

    def delete_all(self):
        for key in self.keys():
            self.delete(key)


hs_redis = RedisCache()
